<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\EmailTemplate;
use App\Models\GeneralSetting;
use App\Helpers\EmailHelper;
use Illuminate\Http\Request;

class EmailTemplateController extends Controller
{
    public function emailConfig()
    {
        $data['pageTitle'] = 'Email Configuration';
        $data['navEmailManagerActiveClass'] = 'active';
        $data['subNavEmailConfigActiveClass'] = 'active';

        return view('backend.email.config')->with($data);
    }

    public function emailConfigUpdate(Request $request)
    {
        $data = $request->validate([
            'site_email' => 'required|email',
            'email_method' => 'required',
            'email_config' => "required_if:email_method,==,smtp",
            'email_config.*' => 'required_if:email_method,==,smtp'
        ]);

        $general = GeneralSetting::first();

        // Handle admin notifications
        $adminNotifications = $request->admin_notifications ?? [];
        
        // Process emails from textarea
        if (isset($adminNotifications['emails'])) {
            $emails = array_filter(array_map('trim', explode("\n", $adminNotifications['emails'])));
            $adminNotifications['emails'] = $emails;
        }
        
        // Ensure boolean values
        $adminNotifications['enabled'] = isset($adminNotifications['enabled']);
        if (isset($adminNotifications['notifications'])) {
            $adminNotifications['notifications']['deposits'] = isset($adminNotifications['notifications']['deposits']);
            $adminNotifications['notifications']['withdrawals'] = isset($adminNotifications['notifications']['withdrawals']);
            $adminNotifications['notifications']['investments'] = isset($adminNotifications['notifications']['investments']);
        }

        // Update data with admin notifications
        $data['admin_notifications'] = json_encode($adminNotifications);
        $general->update($data);

        $notify[] = ['success', "Email Configuration Updated Successfully"];

        return redirect()->back()->withNotify($notify);
    }

    public function emailTemplates()
    {
        $data['pageTitle'] = 'Email Templates';
        $data['navEmailManagerActiveClass'] = 'active';
        $data['subNavEmailTemplatesActiveClass'] = 'active';

        $data['emailTemplates'] = EmailTemplate::latest()->paginate();

        return view('backend.email.templates')->with($data);
    }

    public function emailTemplatesEdit(EmailTemplate $template)
    {
        $pageTitle = 'Template Edit';

        return view('backend.email.edit', compact('pageTitle', 'template'));
    }

    public function emailTemplatesUpdate(Request $request, EmailTemplate $template)
    {
        $data = $request->validate([
            'subject' => 'required',
            'template' => 'required'
        ]);

        $template->update($data);

        $notify[] = ['success', "Email Template Updated Successfully"];

        return redirect()->back()->withNotify($notify);
    }

    public function previewTemplate(EmailTemplate $template)
    {
        try {
            $preview = EmailHelper::previewEmail($template->name);
            return response($preview);
        } catch (\Exception $e) {
            return response('Template preview not available: ' . $e->getMessage(), 404);
        }
    }

    public function updateModernTemplates()
    {
        try {
            EmailHelper::updateEmailTemplates();
            $notify[] = ['success', "Email templates updated to modern design successfully"];
        } catch (\Exception $e) {
            $notify[] = ['error', "Failed to update templates: " . $e->getMessage()];
        }

        return redirect()->back()->withNotify($notify);
    }
}
