<?php

namespace App\Helpers;

use App\Models\EmailTemplate;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\View;

class EmailHelper
{
    /**
     * Send email using modern template system
     */
    public static function sendTemplatedEmail($templateName, $recipient, $data = [])
    {
        // Get template from database
        $emailTemplate = EmailTemplate::where('name', $templateName)->first();
        
        if (!$emailTemplate) {
            throw new \Exception("Email template '{$templateName}' not found");
        }

        // Map template names to view files
        $templateMap = [
            'PASSWORD_RESET' => 'emails.password-reset',
            'PAYMENT_SUCCESSFULL' => 'emails.payment-confirmation',
            'PAYMENT_CONFIRMED' => 'emails.payment-confirmation',
            'VERIFY_EMAIL' => 'emails.email-verification',
            'WITHDRAWAL_UPDATE' => 'emails.withdrawal-update',
            'GENERAL_NOTIFICATION' => 'emails.general-notification',
        ];

        $viewName = $templateMap[$templateName] ?? 'emails.general-notification';

        // Prepare email data
        $emailData = array_merge([
            'subject' => $emailTemplate->subject,
            'username' => $recipient['name'] ?? 'Valued Client',
            'sent_from' => 'Fortis Beacon Holdings',
        ], $data);

        // Send email
        Mail::send($viewName, $emailData, function ($message) use ($recipient, $emailData) {
            $message->to($recipient['email'], $recipient['name'] ?? '')
                   ->subject($emailData['subject']);
        });

        return true;
    }

    /**
     * Update existing email templates with modern design
     */
    public static function updateEmailTemplates()
    {
        $templates = [
            [
                'name' => 'PASSWORD_RESET',
                'subject' => 'Password Reset Verification Code',
                'template' => 'Modern responsive template for password reset',
                'meaning' => json_encode([
                    'username' => 'Email Receiver Name',
                    'code' => 'Password Reset Verification Code',
                    'sent_from' => 'Email Sent From'
                ])
            ],
            [
                'name' => 'PAYMENT_SUCCESSFULL',
                'subject' => 'Payment Confirmation - Investment Activated',
                'template' => 'Modern responsive template for payment confirmation',
                'meaning' => json_encode([
                    'username' => 'Email Receiver Name',
                    'trx' => 'Transaction Number',
                    'amount' => 'Payment Amount',
                    'plan' => 'Investment Plan Name',
                    'currency' => 'Payment Currency',
                    'charge' => 'Processing Fee',
                    'sent_from' => 'Email Sent From'
                ])
            ],
            [
                'name' => 'VERIFY_EMAIL',
                'subject' => 'Verify Your Email Address - Welcome to Fortis Beacon Holdings',
                'template' => 'Modern responsive template for email verification',
                'meaning' => json_encode([
                    'username' => 'Email Receiver Name',
                    'code' => 'Email Verification Code',
                    'sent_from' => 'Email Sent From'
                ])
            ],
            [
                'name' => 'WITHDRAWAL_UPDATE',
                'subject' => 'Important Update Regarding Your Withdrawal Request',
                'template' => 'Modern responsive template for withdrawal updates',
                'meaning' => json_encode([
                    'username' => 'Email Receiver Name',
                    'status' => 'Withdrawal Status',
                    'transaction_details' => 'Transaction Information',
                    'sent_from' => 'Email Sent From'
                ])
            ]
        ];

        foreach ($templates as $template) {
            EmailTemplate::updateOrCreate(
                ['name' => $template['name']],
                $template
            );
        }

        return true;
    }

    /**
     * Get email preview for testing
     */
    public static function previewEmail($templateName, $sampleData = [])
    {
        $templateMap = [
            'PASSWORD_RESET' => 'emails.password-reset',
            'PAYMENT_SUCCESSFULL' => 'emails.payment-confirmation',
            'VERIFY_EMAIL' => 'emails.email-verification',
            'WITHDRAWAL_UPDATE' => 'emails.withdrawal-update',
        ];

        $viewName = $templateMap[$templateName] ?? 'emails.general-notification';

        // Sample data for preview
        $defaultData = [
            'username' => 'John Doe',
            'code' => '123456',
            'trx' => 'TXN789123456',
            'amount' => '5,000.00',
            'currency' => 'USD',
            'plan' => 'Premium Investment Plan',
            'charge' => '25.00',
            'status' => 'approved',
            'transaction_details' => [
                'id' => 'WD789123456',
                'amount' => '2,500.00',
                'currency' => 'USD',
                'date' => 'December 15, 2024',
                'processing_time' => '1-3 business days'
            ],
            'sent_from' => 'Fortis Beacon Holdings',
            'subject' => 'Email Preview'
        ];

        $data = array_merge($defaultData, $sampleData);

        return View::make($viewName, $data)->render();
    }

    /**
     * Validate email template data
     */
    public static function validateTemplateData($templateName, $data)
    {
        $requiredFields = [
            'PASSWORD_RESET' => ['username', 'code'],
            'PAYMENT_SUCCESSFULL' => ['username', 'trx', 'amount', 'plan'],
            'VERIFY_EMAIL' => ['username', 'code'],
            'WITHDRAWAL_UPDATE' => ['username', 'status']
        ];

        $required = $requiredFields[$templateName] ?? [];
        $missing = [];

        foreach ($required as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                $missing[] = $field;
            }
        }

        if (!empty($missing)) {
            throw new \Exception("Missing required fields: " . implode(', ', $missing));
        }

        return true;
    }
}
